<?php
if (session_status() === PHP_SESSION_NONE) session_start();
require_once __DIR__ . '/../../Config/config.php';
require_once __DIR__ . '/../../Core/Database.php';

if (empty($_SESSION['user'])) {
  header('Location: ' . BASE_URL . 'login.php');
  exit;
}

$pdo = Database::db();
$user = $_SESSION['user'] ?? ['username' => 'Agent'];

// Stats
function getStats($pdo) {
  return [
    'calls' => $pdo->query("SELECT COUNT(*) AS c FROM call_history")->fetch()['c'] ?? 0,
    'customers' => $pdo->query("SELECT COUNT(*) AS c FROM customers")->fetch()['c'] ?? 0,
    'tickets' => $pdo->query("SELECT COUNT(*) AS c FROM tickets")->fetch()['c'] ?? 0,
    'performance' => rand(80, 99)
  ];
}
$stats = getStats($pdo);

// Agent status
$statusRow = null;
if (!empty($user['id'])) {
  $stmt = $pdo->prepare("SELECT status FROM agent_status WHERE user_id = ?");
  $stmt->execute([$user['id']]);
  $statusRow = $stmt->fetch();
}
$agentStatus = $statusRow['status'] ?? 'offline';
$statusColors = [
  'available' => ['🟢', 'Connected', 'pulse-green'],
  'busy'      => ['🔴', 'Busy', 'pulse-red'],
  'break'     => ['🟡', 'On Break', 'pulse-yellow'],
  'offline'   => ['⚫', 'Offline', '']
];
$badge = $statusColors[$agentStatus] ?? $statusColors['offline'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>CyberWall CallCenter - Dashboard</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.3/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    body { background-color: #f8f9fa; font-family: "Segoe UI", sans-serif; }
    .sidebar { position: fixed; top:0; left:0; width:240px; height:100vh; background:#1f2d3a; color:#fff; display:flex; flex-direction:column; justify-content:space-between; }
    .sidebar h4 { padding:20px; text-align:center; border-bottom:1px solid rgba(255,255,255,0.1); }
    .sidebar .nav-link { color:#cfd8dc; padding:10px 20px; }
    .sidebar .nav-link.active,.sidebar .nav-link:hover { background:#273746; color:#fff; }
    .main { margin-left:240px; padding:2rem; }
    .card { border:none; border-radius:10px; box-shadow:0 3px 6px rgba(0,0,0,0.1); }

    /* Pulse animation */
    .pulse { display:inline-flex; align-items:center; justify-content:center; padding:4px 10px; border-radius:20px; font-weight:600; position:relative; overflow:hidden; }
    .pulse::before { content:""; position:absolute; inset:0; border-radius:inherit; animation:pulseAnim 1.6s infinite; opacity:0.6; }
    @keyframes pulseAnim { 0% { transform:scale(1); opacity:0.7; } 50% { transform:scale(1.3); opacity:0.2; } 100% { transform:scale(1); opacity:0.7; } }
    .pulse-green::before { background:rgba(40,167,69,0.3); }
    .pulse-red::before { background:rgba(220,53,69,0.3); }
    .pulse-yellow::before { background:rgba(255,193,7,0.3); }
    .pulse span { position:relative; z-index:2; }

    /* --- Friendly Dial Pad --- */
    .softphone { max-width: 720px; margin: 0 auto; }
    #dialNumber {
      background:#fff; border:1px solid #e5e7eb; border-radius:10px;
      padding:14px 16px; box-shadow:0 2px 4px rgba(0,0,0,.04);
    }
    .dial-grid { gap:10px; }
    .dial-key {
      width:88px; height:88px; border-radius:16px; border:1px solid #dfe3e7;
      background:#fff; display:flex; flex-direction:column; align-items:center; justify-content:center;
      font-weight:700; font-size:1.6rem; color:#222; line-height:1; user-select:none; cursor:pointer;
      box-shadow:0 2px 6px rgba(0,0,0,.06); transition:transform .08s ease, background .2s ease, box-shadow .2s ease;
    }
    .dial-key small { margin-top:2px; font-size:.68rem; color:#8c96a0; font-weight:600; letter-spacing:.5px; }
    .dial-key:hover { background:#f6f7f9; box-shadow:0 4px 10px rgba(0,0,0,.08); }
    .dial-key:active { transform:scale(.96); }
    .dial-actions .round {
      width:86px; height:86px; border-radius:50%; display:flex; align-items:center; justify-content:center;
      box-shadow:0 8px 20px rgba(0,0,0,.12); transition:transform .08s ease, box-shadow .2s ease, filter .2s ease;
    }
    .dial-actions .round:active { transform:scale(.96); }
    .dial-actions .call { background:#28a745; color:#fff; }
    .dial-actions .hang { background:#dc3545; color:#fff; }
  </style>
</head>
<body>

<!-- Sidebar -->
<div class="sidebar">
  <div>
    <h4><i class="bi bi-telephone-fill me-2"></i>CyberWall</h4>
    <nav class="nav flex-column">
      <a class="nav-link active" href="#"><i class="bi bi-speedometer2 me-2"></i> Dashboard</a>
      <a class="nav-link" href="#"><i class="bi bi-people me-2"></i> Customers</a>
      <a class="nav-link" href="#"><i class="bi bi-telephone me-2"></i> Calls</a>
      <a class="nav-link" href="#"><i class="bi bi-ticket-detailed me-2"></i> Tickets</a>
      <a class="nav-link" href="#"><i class="bi bi-bar-chart-line me-2"></i> Reports</a>
    </nav>
  </div>
  <div><a href="login.php" class="nav-link text-danger"><i class="bi bi-box-arrow-right me-2"></i> Logout</a></div>
</div>

<!-- Main -->
<div class="main">
  <div class="d-flex justify-content-between align-items-center mb-4">
    <h3 class="fw-bold mb-0">Welcome, <?= htmlspecialchars($user['display_name'] ?? $user['username'] ?? 'Agent') ?></h3>
    <div class="d-flex align-items-center gap-3">
      <span id="sipStatus" class="fw-semibold text-secondary">⚫ Disconnected</span>
      <button id="btnRegister" class="btn btn-outline-success btn-sm">Register</button>
      <button id="btnUnregister" class="btn btn-outline-danger btn-sm" disabled>Unregister</button>
      <div id="agentStatus" class="pulse <?= $badge[2] ?>"><span><?= $badge[0] . ' ' . $badge[1] ?></span></div>
      <div class="btn-group btn-group-sm">
        <button class="btn btn-outline-success" onclick="setAgent('available')">Connected</button>
        <button class="btn btn-outline-warning" onclick="setAgent('break')">On Break</button>
        <button class="btn btn-outline-danger" onclick="setAgent('busy')">Busy</button>
      </div>
    </div>
  </div>

  <!-- Stats -->
  <div class="row g-4 mb-4">
    <div class="col-md-3"><div class="card text-center p-3"><i class="bi bi-telephone-forward display-6 text-primary"></i><h6>Active Calls</h6><h3><?= $stats['calls'] ?></h3></div></div>
    <div class="col-md-3"><div class="card text-center p-3"><i class="bi bi-people display-6 text-success"></i><h6>Customers</h6><h3><?= $stats['customers'] ?></h3></div></div>
    <div class="col-md-3"><div class="card text-center p-3"><i class="bi bi-ticket-perforated display-6 text-warning"></i><h6>Tickets</h6><h3><?= $stats['tickets'] ?></h3></div></div>
    <div class="col-md-3"><div class="card text-center p-3"><i class="bi bi-bar-chart-line display-6 text-danger"></i><h6>Performance</h6><h3><?= $stats['performance'] ?>%</h3></div></div>
  </div>

  <!-- Friendly Dial Pad -->
  <div class="card p-4 mb-5 text-center softphone">
    <h5 class="mb-3 fw-semibold"><i class="bi bi-telephone me-2 text-primary"></i>Softphone</h5>

    <input id="dialNumber" type="text" class="form-control text-center fs-4 fw-bold mb-3"
           placeholder="Tap a number..." readonly style="letter-spacing:2px;">

    <div class="d-flex flex-column align-items-center dial-grid">
      <div class="d-flex">
        <button class="dial-key" onclick="addDigit('1')">1</button>
        <button class="dial-key" onclick="addDigit('2')">2<small>ABC</small></button>
        <button class="dial-key" onclick="addDigit('3')">3<small>DEF</small></button>
      </div>
      <div class="d-flex">
        <button class="dial-key" onclick="addDigit('4')">4<small>GHI</small></button>
        <button class="dial-key" onclick="addDigit('5')">5<small>JKL</small></button>
        <button class="dial-key" onclick="addDigit('6')">6<small>MNO</small></button>
      </div>
      <div class="d-flex">
        <button class="dial-key" onclick="addDigit('7')">7<small>PQRS</small></button>
        <button class="dial-key" onclick="addDigit('8')">8<small>TUV</small></button>
        <button class="dial-key" onclick="addDigit('9')">9<small>WXYZ</small></button>
      </div>
      <div class="d-flex">
        <button class="dial-key" onclick="addDigit('*')">*</button>
        <button class="dial-key" onclick="addDigit('0')">0</button>
        <button class="dial-key" onclick="addDigit('#')">#</button>
      </div>
    </div>

    <div class="d-flex justify-content-center mt-4 gap-4 dial-actions">
      <button id="callBtn" class="round call" onclick="makeCall()" title="Call">
        <i class="bi bi-telephone-fill fs-3"></i>
      </button>
      <button id="hangupBtn" class="round hang" onclick="hangupCall()" title="Hang up">
        <i class="bi bi-telephone-x-fill fs-3"></i>
      </button>
    </div>
  </div>
</div>

<!-- SIP.js -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/sip.js/0.20.0/sip.min.js"></script>
<script>
let userAgent, currentSession = null;

async function setAgent(status) {
  await fetch('<?= BASE_URL ?>sip_status_update.php', {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: 'status=' + encodeURIComponent(status)
  });
}

function addDigit(d) {
  const input = document.getElementById('dialNumber');
  input.value += d;
  input.classList.add('pulse');
  setTimeout(() => input.classList.remove('pulse'), 150);
}

function initSIP() {
  try {
    const uri = SIP.UserAgent.makeURI("sip:5391005@pbx.cyberwallvoip.com");

    userAgent = new SIP.UserAgent({
      uri: uri,
      transportOptions: { server: 'wss://pbx.cyberwallvoip.com:8089/ws' },
      authorizationUsername: '5391005',
      authorizationPassword: '5hfmgdqqaJwe4euBFXEf5cgUf',
      displayName: 'Agent',
      sessionDescriptionHandlerFactoryOptions: {
        constraints: { audio: true, video: false },
        peerConnectionConfiguration: {
          iceServers: [{ urls: 'stun:stun.l.google.com:19302' }]
        }
      }
    });

    const registerer = new SIP.Registerer(userAgent);
    const statusEl = document.getElementById('sipStatus');

    userAgent.start().then(() => {
      registerer.register();
      statusEl.textContent = '🟢 Registered';
      statusEl.className = 'fw-semibold text-success';
      setAgent('available');
    });

    userAgent.delegate = {
      onInvite: (invitation) => {
        console.log("📞 Incoming call from", invitation.remoteIdentity.uri.toString());
        invitation.accept();
        statusEl.textContent = '🔵 Incoming Call...';
        statusEl.className = 'fw-semibold text-warning';
        setAgent('busy');

        invitation.sessionDescriptionHandler.on('addTrack', (e) => {
          const audio = document.createElement('audio');
          audio.srcObject = e.streams[0];
          audio.autoplay = true;
          document.body.appendChild(audio);
        });

        invitation.stateChange.addListener((state) => {
          if (state === SIP.SessionState.Terminated) {
            statusEl.textContent = '🟢 Registered';
            statusEl.className = 'fw-semibold text-success';
            setAgent('available');
          }
        });
      }
    };

    document.getElementById('btnRegister').disabled = true;
    document.getElementById('btnUnregister').disabled = false;

  } catch (e) {
    console.error("❌ SIP init error", e);
    document.getElementById('sipStatus').textContent = '❌ SIP Error';
  }
}

function makeCall() {
  const num = document.getElementById('dialNumber').value.trim();
  if (!num) return alert('Enter a number');
  const targetUri = SIP.UserAgent.makeURI(`sip:${num}@pbx.cyberwallvoip.com`);
  const inviter = new SIP.Inviter(userAgent, targetUri);
  inviter.invite();

  document.getElementById('sipStatus').textContent = '📞 Calling...';
  setAgent('busy');

  inviter.stateChange.addListener((state) => {
    if (state === SIP.SessionState.Terminated) {
      document.getElementById('sipStatus').textContent = '🟢 Registered';
      setAgent('available');
    }
  });

  currentSession = inviter;
}

function hangupCall() {
  if (currentSession) {
    currentSession.dispose();
    currentSession = null;
  }
  document.getElementById('sipStatus').textContent = '🟢 Registered';
  setAgent('available');
}

function stopSIP() {
  userAgent.stop();
  document.getElementById('sipStatus').textContent = '⚫ Disconnected';
  setAgent('offline');
  document.getElementById('btnRegister').disabled = false;
  document.getElementById('btnUnregister').disabled = true;
}

document.getElementById('btnRegister').addEventListener('click', initSIP);
document.getElementById('btnUnregister').addEventListener('click', stopSIP);
</script>
</body>
</html>
